import argparse
from pathlib import Path
from typing import List

import fq_utils
import runner
from runner import ProcessCmd


def create_somatic_cmd(static_args: str, normal_fastqs: List[str], tumor_fastqs: List[str]) -> ProcessCmd:
    # Process and add FASTQ commands
    tumor_sample_names, tumor_fastq_list = fq_utils.process_fastqs(tumor_fastqs, Path('.'), 'tumor-fastq-list.csv')
    normal_sample_names, normal_fastq_list = fq_utils.process_fastqs(normal_fastqs, Path('.'))

    # We can only take one tumor normal pair because it is too restrictive to match pairs from naming alone
    if len(tumor_sample_names) > 1 or len(normal_sample_names) > 1:
        raise ValueError('Only one sample can be submitted at a time, and the files must match the convention, ' +
                         '(.+_S[0-9]+)_L([0-9]{3})_R([1-2])_([0-9]{3})$ where sample name is group 1.')

    cmd_str = ' '.join([
        static_args,
        f'--tumor-fastq-list {tumor_fastq_list}',
        f'--tumor-fastq-list-sample-id {tumor_sample_names[0]}',
    ])

    if normal_sample_names:
        cmd_str = ' '.join([
            cmd_str,
            f'--fastq-list {normal_fastq_list}',
            f'--fastq-list-sample-id {normal_sample_names[0]}',
        ])
    return ProcessCmd(cmd_str)


def execute_somatic(ref_tar: str, dragen_args: List[str], tumor_fastqs: List[str], normal_fastqs: List[str]):
    process_cmds = runner.create_setup_ref_cmds(ref_tar)
    static_dragen_args = runner.create_dragen_cmd(dragen_args)
    process_cmds.append(create_somatic_cmd(static_dragen_args, normal_fastqs, tumor_fastqs))
    runner.run_dragen_cmds(process_cmds)


def main():
    parser = argparse.ArgumentParser()
    parser.add_argument('--normal-fastqs', nargs='*', default=[], type=str)
    parser.add_argument('--tumor-fastqs', nargs='*', default=[], type=str)
    parser.add_argument('--ref-tar', type=str)
    parser.add_argument('--debug', action='store_true')
    args, extras = parser.parse_known_args()
    runner.execute_with_no_traceback(
        lambda: execute_somatic(args.ref_tar, extras, args.tumor_fastqs, args.normal_fastqs),
        args.debug
    )


if __name__ == '__main__':
    main()
