from typing import List
from unittest.mock import MagicMock, patch

import batch_runner
from runner import ProcessCmd
from test_utils import assert_args


@patch('builtins.open')
def test_when_multiple_fastqs_passed__correctly_invokes_dragen(mock_open):
    batch_runner.runner.run_dragen_cmds = MagicMock()
    batch_runner.execute_batch('/my/folder/reference.tar', ['--foo', 'bar', '--baz', 'abc'], [
        '/my/folder/s01-NFE-CEX-NA12878_S1_L001_R1_001.fastq.gz',
        '/my/folder/s01-NFE-CEX-NA12878_S1_L001_R2_001.fastq.gz',
        '/my/folder/s01-NFE-CEX-NA12878_S1_L002_R1_001.fastq.gz',
        '/my/folder/s01-NFE-CEX-NA12878_S1_L002_R2_001.fastq.gz',
        '/my/folder/s02-NFE-CEX-NA12878_S13_L001_R1_001.fastq.gz',
        '/my/folder/s02-NFE-CEX-NA12878_S13_L001_R2_001.fastq.gz',
        '/my/folder/s02-NFE-CEX-NA12878_S13_L002_R1_001.fastq.gz',
        '/my/folder/s02-NFE-CEX-NA12878_S13_L002_R2_001.fastq.gz',
    ], [], None)

    setup_cmds: List[ProcessCmd] = batch_runner.runner.run_dragen_cmds.call_args_list[0].args[0]
    assert len(setup_cmds) == 2
    assert setup_cmds[1].cmd == 'tar -C /ephemeral/reference/index -xf /my/folder/reference.tar'
    assert setup_cmds[1].working_dir == '/ephemeral/reference/index'

    cmds: List[ProcessCmd] = batch_runner.runner.run_dragen_cmds.call_args_list[1].args[0]
    assert_args([
        '--output-directory .',
        '--foo bar ',
        '--baz abc',
        '--output-file-prefix s01-NFE-CEX-NA12878_S1',
        '--vc-sample-name s01-NFE-CEX-NA12878_S1',
        '--fastq-list-sample-id s01-NFE-CEX-NA12878_S1',
    ], cmds[0].cmd)
    assert_args([
        '--foo bar ',
        '--baz abc ',
        '--output-file-prefix s02-NFE-CEX-NA12878_S13',
        '--output-directory .',
        '--vc-sample-name s02-NFE-CEX-NA12878_S13',
        '--fastq-list-sample-id s02-NFE-CEX-NA12878_S13',
    ], cmds[1].cmd)


@patch('builtins.open')
def test_when_cnv_normals_dir_passed__correctly_appends_args(mock_open):
    batch_runner.glob.glob = MagicMock(return_value=['normal1', 'normal2'])
    cmds = batch_runner.create_batch_cmds('--foo bar', [], ['one.bam'], '/ephemeral/cnv-normals/')
    assert len(cmds) == 1
    assert ('--foo bar --output-file-prefix one '
            '--bam-input one.bam --cnv-normals-file normal1 --cnv-normals-file normal2') in cmds[0].cmd
    batch_runner.glob.glob.assert_called_with('/ephemeral/cnv-normals/*')


@patch('builtins.open')
def test_when_no_cnv_normals_dir_passed__skips_arg(mock_open):
    cmds = batch_runner.create_batch_cmds('--foo bar', [], ['one.bam'], None)
    assert len(cmds) == 1
    assert ('--foo bar --output-file-prefix one '
            '--bam-input one.bam') in cmds[0].cmd
