from typing import Iterable

import fq_utils


def test_when_fastq_filename_passed__correctly_parses_file():
    files = [
        '/my/favorite/folder/s01-NFE-CEX-NA12878_S1_L002_R1_001.fastq.gz',
        's02.NFE.CEX.NA12878_S13_L001_R2_001.fq.gz',
        '/data/cwl-stage-dir/stg0c9f5919-6fe3-46f9-b24a-086e9148244e/HM07-HCC1187BL-2b-A012_S6_L001_R1_002.fastq.gz',
    ]
    results = []
    expected = [
        ('s01-NFE-CEX-NA12878_S1', 2, 1, 's01-NFE-CEX-NA12878_S1_L002_001'),
        ('s02.NFE.CEX.NA12878_S13', 1, 2, 's02.NFE.CEX.NA12878_S13_L001_001'),
        ('HM07-HCC1187BL-2b-A012_S6', 1, 1, 'HM07-HCC1187BL-2b-A012_S6_L001_002'),
    ]
    for file in files:
        results.append(fq_utils.parse_fastq(file))
    assert results == expected


def test_when_fastqs_passed__correctly_constructs_samples():
    files = [
        '/my/folder/s01-NFE-CEX-NA12878_S1_L001_R1_001.fastq.gz',
        '/my/folder/s01-NFE-CEX-NA12878_S1_L001_R2_001.fastq.gz',
        '/my/folder/s01-NFE-CEX-NA12878_S1_L002_R1_001.fastq.gz',
        '/my/folder/s01-NFE-CEX-NA12878_S1_L002_R2_001.fastq.gz',
        '/my/folder/s02-NFE-CEX-NA12878_S13_L001_R1_001.fastq.gz',
        '/my/folder/s02-NFE-CEX-NA12878_S13_L001_R2_001.fastq.gz',
        '/my/folder/s02-NFE-CEX-NA12878_S13_L002_R1_001.fastq.gz',
        '/my/folder/s02-NFE-CEX-NA12878_S13_L002_R2_001.fastq.gz',
    ]
    samples = fq_utils.construct_samples(files)
    assert len(samples) == 2
    assert len(samples['s01-NFE-CEX-NA12878_S1']) == 2
    assert len(samples['s02-NFE-CEX-NA12878_S13']) == 2


def test_when_fastqs_passed__correctly_constructs_rgid():
    files = [
        '/Illumina Public DRAGEN Methylation Data/DRAGEN Methylation Input/HM07-HCC1187BL-2b-A012_S6_L001_R1_001.fastq.gz',
        '/Illumina Public DRAGEN Methylation Data/DRAGEN Methylation Input/HM07-HCC1187BL-2b-A012_S6_L001_R1_002.fastq.gz',
        '/Illumina Public DRAGEN Methylation Data/DRAGEN Methylation Input/HM07-HCC1187BL-2b-A012_S6_L001_R2_001.fastq.gz',
        '/Illumina Public DRAGEN Methylation Data/DRAGEN Methylation Input/HM07-HCC1187BL-2b-A012_S6_L001_R2_002.fastq.gz',
        '/Illumina Public DRAGEN Methylation Data/DRAGEN Methylation Input/HM07-HCC1187BL-2b-A012_S6_L002_R1_001.fastq.gz',
        '/Illumina Public DRAGEN Methylation Data/DRAGEN Methylation Input/HM07-HCC1187BL-2b-A012_S6_L002_R1_002.fastq.gz',
        '/Illumina Public DRAGEN Methylation Data/DRAGEN Methylation Input/HM07-HCC1187BL-2b-A012_S6_L002_R2_001.fastq.gz',
        '/Illumina Public DRAGEN Methylation Data/DRAGEN Methylation Input/HM07-HCC1187BL-2b-A012_S6_L002_R2_002.fastq.gz',
    ]
    samples = fq_utils.construct_samples(files)
    assert len(samples) == 1
    assert len(samples['HM07-HCC1187BL-2b-A012_S6']) == 4
    assert has_single(
        fq for fq in samples['HM07-HCC1187BL-2b-A012_S6'] if fq.rgid == 'HM07-HCC1187BL-2b-A012_S6_L001_001'
    )
    assert has_single(
        fq for fq in samples['HM07-HCC1187BL-2b-A012_S6'] if fq.rgid == 'HM07-HCC1187BL-2b-A012_S6_L001_002'
    )
    assert has_single(
        fq for fq in samples['HM07-HCC1187BL-2b-A012_S6'] if fq.rgid == 'HM07-HCC1187BL-2b-A012_S6_L002_001'
    )
    assert has_single(
        fq for fq in samples['HM07-HCC1187BL-2b-A012_S6'] if fq.rgid == 'HM07-HCC1187BL-2b-A012_S6_L002_002'
    )


def has_single(iterable: Iterable) -> bool:
    """Checks interable to see if there is only a single truthy value"""
    i = iter(iterable)
    #  Consumes i until first truthy value, then keeps going to check if there is another truthy value
    return any(i) and not any(i)
